// Pom1 Apple 1 Emulator
// Copyright (C) 2000 Verhille Arnaud
// Copyright (C) 2006 John D. Corrado
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

#include <SDL.h>
#include <string.h>
#include "screen.h"

static unsigned char charac[1024], screenTbl[960];
static int indexX, indexY, pixelSize = 2, _scanline = 0, terminalSpeed = 60;
static long lastTime;
static int _fullscreen = 0;

int loadCharMap(void)
{
	FILE *fd = fopen("rom/charmap.rom", "rb");

	if (fd)
	{
		fread(charac, 1, 1024, fd);
		fclose(fd);
	}
	else
		return 0;

	return 1;
}

void resetScreen(void)
{
	lastTime = SDL_GetTicks();

	indexX = indexY = 0;

	memset(screenTbl, 0, 960);

	updateScreen();
}


void setPixelSize(int ps)
{
	pixelSize = ps;
}

int getPixelSize(void)
{
	return pixelSize;
}

void setScanline(int scanline)
{
	_scanline = scanline;
}

int getScanline(void)
{
	return _scanline;
}

void setTerminalSpeed(int ts)
{
	terminalSpeed = ts;
}

int getTerminalSpeed(void)
{
	return terminalSpeed;
}

static void synchronizeOutput(void)
{
	int sleepMillis = (int)((1000 / terminalSpeed) - (SDL_GetTicks() - lastTime));

	if (sleepMillis > 0)
		SDL_Delay(sleepMillis);

	lastTime = SDL_GetTicks();
}

static void newLine(void)
{
	int i;

	for (i = 0; i < 23; ++i)
		memcpy(&screenTbl[i * 40], &screenTbl[(i + 1) * 40], 40);

	memset(&screenTbl[920], 0, 40);
}

void outputDsp(unsigned char dsp)
{
	switch (dsp)
	{
	case 0x5F:
		if (indexX == 0)
		{
			indexY--;
			indexX = 39;
		}
		else
			indexX--;

		screenTbl[indexY * 40 + indexX] = 0;
		break;
	case 0x0A:
	case 0x0D:
		indexX = 0;
		indexY++;
		break;
	case 0x00:
	case 0x7F:
		break;
	default:
		screenTbl[indexY * 40 + indexX] = dsp;
		indexX++;
		break;
	}

	if (indexX == 40)
	{
		indexX = 0;
		indexY++;
	}
	if (indexY == 24)
	{
		newLine();
		indexY--;
	}

	updateScreen();

	synchronizeOutput();
}

static void drawCharac(int xPosition, int yPosition, unsigned char r, unsigned char g, unsigned char b, unsigned char characNumber)
{
	SDL_Rect rect;
	int k, l;

	for (k = 0; k < 8; ++k)
	{
		for (l = 1; l < 8; ++l)
		{
			if (charac[characNumber * 8 + k] & (0x01 << l))
			{
				rect.x = xPosition + pixelSize * (l - 1);
				rect.y = yPosition + pixelSize * k;
				rect.w = pixelSize;
				rect.h = pixelSize - (_scanline ? 1 : 0);

				SDL_FillRect(SDL_GetVideoSurface(), &rect, SDL_MapRGB(SDL_GetVideoSurface()->format, r, g, b));
			}
		}
	}
}

void updateScreen(void)
{
	int xPosition, yPosition;
	int i, j;

	SDL_FillRect(SDL_GetVideoSurface(), NULL, 0);

	for (i = 0; i < 40; ++i)
	{
		for (j = 0; j < 24; ++j)
		{
			xPosition = i * pixelSize * 7;
			yPosition = j * pixelSize * 8;

			drawCharac(xPosition, yPosition, 0, 255, 0, screenTbl[j * 40 + i]);
		}
	}

	drawCharac(indexX * pixelSize * 7, indexY * pixelSize * 8, 0, 255, 0, 0x01);

	SDL_Flip(SDL_GetVideoSurface());
}

void drawCharacter(int xPosition, int yPosition, unsigned char r, unsigned char g, unsigned char b, unsigned char characNumber)
{
	if (_scanline)
	{
		_scanline = 0;
		drawCharac(xPosition, yPosition, 0, 0, 0, characNumber);
		_scanline = 1;
	}
	else
		drawCharac(xPosition, yPosition, 0, 0, 0, characNumber);
}

void setFullScreen(int fullscreen)
{
	_fullscreen = fullscreen;
}

int getFullScreen()
{
	return _fullscreen;
}
